# --- raw_converter_interactive_v2.py ---

import os
import sys
import rawpy
from PIL import Image
import multiprocessing
import time
import glob
from tqdm import tqdm

# ===========================================================
#               【核心功能函数 - 无需修改】
# ===========================================================

def convert_raw_to_jpg(raw_path, output_dir, quality, input_root_dir, search_subdirs):
    """转换单个 RAW 文件并返回处理结果信息"""
    try:
        filename = os.path.basename(raw_path)
        base_name = os.path.splitext(filename)[0]
        
        if search_subdirs and os.path.dirname(raw_path) != input_root_dir:
            relative_dir = os.path.relpath(os.path.dirname(raw_path), input_root_dir)
            target_dir = os.path.join(output_dir, relative_dir)
            os.makedirs(target_dir, exist_ok=True)
            jpg_path = os.path.join(target_dir, f"{base_name}.jpg")
        else:
            jpg_path = os.path.join(output_dir, f"{base_name}.jpg")

        if os.path.exists(jpg_path):
            return f"跳过: {filename}"

        with rawpy.imread(raw_path) as raw:
            rgb_data = raw.postprocess(use_camera_wb=True, output_bps=8, no_auto_bright=True)
            pil_image = Image.fromarray(rgb_data)
            pil_image.save(jpg_path, 'JPEG', quality=quality, optimize=True)
        return f"成功: {filename}"
            
    except Exception as e:
        return f"失败: {os.path.basename(raw_path)} - {e}"

def convert_raw_to_jpg_starmap_helper(args):
    """辅助函数"""
    return convert_raw_to_jpg(*args)

def batch_convert_raws(input_dir, output_dir, quality, search_subdirs, num_processes):
    """主函数"""
    start_time = time.time()
    os.makedirs(output_dir, exist_ok=True)

    print("\n🚀 正在扫描 RAW 文件...")
    raw_extensions = ['*.CR2', '*.CR3', '*.NEF', '*.ARW', '*.DNG', '*.PEF', '*.RAF', '*.RW2', '*.ORF']
    raw_files = []
    for ext in raw_extensions:
        search_pattern = os.path.join(input_dir, '**', ext) if search_subdirs else os.path.join(input_dir, ext)
        raw_files.extend(glob.glob(search_pattern, recursive=search_subdirs))
    
    if not raw_files:
        print(f"❌ 在目录 '{input_dir}' 中没有找到支持的 RAW 文件。请检查路径或文件类型。")
        return

    print(f"✅ 共找到 {len(raw_files)} 个 RAW 文件。")
    print(f"🔥 开始使用 {num_processes} 个 CPU 核心进行转换...")

    tasks = [(file, output_dir, quality, input_dir, search_subdirs) for file in raw_files]

    with multiprocessing.Pool(processes=num_processes) as pool:
        results = list(tqdm(pool.imap_unordered(convert_raw_to_jpg_starmap_helper, tasks), total=len(tasks), desc="转换进度"))

    success_count = sum(1 for r in results if r and r.startswith("成功"))
    skipped_count = sum(1 for r in results if r and r.startswith("跳过"))
    failed_count = sum(1 for r in results if r and r.startswith("失败"))

    end_time = time.time()
    print("-" * 50)
    print("🎉 转换完成！")
    print(f"⏱️ 总耗时: {end_time - start_time:.2f} 秒")
    print(f"📊 结果: {success_count} 成功 | {skipped_count} 跳过 | {failed_count} 失败")

# ===========================================================
#               【交互式设置主程序 - 修正版】
# ===========================================================

def get_user_config():
    """通过交互式问答获取用户配置"""
    print("=" * 50)
    print("      欢迎使用极速 RAW to JPG 转换脚本！")
    print("=" * 50)
    
    # 1. 获取 RAW 文件夹路径
    while True:
        raw_input_path = input("\n📁 请将你的【RAW 源文件夹】拖到这里，然后按 Enter: ")
        clean_path = raw_input_path.strip().strip('"')
        if os.path.isdir(clean_path):
            input_dir = clean_path
            break
        else:
            print("❌ 错误: 这不是一个有效的文件夹路径，请重试。")

    # 2. 获取输出文件夹路径
    while True:
        raw_output_path = input("\n📂 请将你的【JPG 输出文件夹】拖到这里，然后按 Enter: ")
        clean_path = raw_output_path.strip().strip('"')
        if clean_path:
            output_dir = clean_path
            break
        else:
            print("❌ 错误: 输出路径不能为空，请重试。")

    # 3. 获取 JPG 质量
    while True:
        try:
            quality_str = input("\n⚙️  请输入 JPG 压缩质量 (1-100, 推荐90, 直接按 Enter 使用默认值): ")
            if not quality_str:
                quality = 90
                break
            quality = int(quality_str)
            if 1 <= quality <= 100:
                break
            else:
                print("❌ 错误: 请输入 1 到 100 之间的数字。")
        except ValueError:
            print("❌ 错误: 无效输入，请输入一个数字。")
            
    # 4. 【新增】询问是否搜索子文件夹
    while True:
        subdirs_choice = input("\n🔍 是否需要处理子文件夹中的文件? (y/n, 直接按 Enter 默认为 '是'): ").lower()
        if subdirs_choice in ['y', 'yes', '']:
            search_subdirs = True
            break
        elif subdirs_choice in ['n', 'no']:
            search_subdirs = False
            break
        else:
            print("❌ 错误: 无效输入，请输入 'y' 或 'n'。")

    return input_dir, output_dir, quality, search_subdirs

if __name__ == '__main__':
    multiprocessing.freeze_support()
    
    try:
        # 获取用户配置（现在会返回4个值）
        input_dir, output_dir, quality, search_subdirs = get_user_config()
        
        # 显示最终配置并请求确认
        print("\n" + "=" * 50)
        print("请确认您的设置:")
        print(f"  -  RAW 来源: {input_dir}")
        print(f"  -  JPG 输出: {output_dir}")
        print(f"  -  JPG 质量: {quality}")
        print(f"  -  处理子文件夹: {'是' if search_subdirs else '否'}") # <-- 新增确认项
        print("=" * 50)
        
        input("✅ 设置无误，请按 Enter 键开始转换...")
        
        # 执行转换
        batch_convert_raws(
            input_dir=input_dir,
            output_dir=output_dir,
            quality=quality,
            search_subdirs=search_subdirs, # <-- 将用户的选择传递给核心函数
            num_processes=multiprocessing.cpu_count()
        )
    
    except KeyboardInterrupt:
        print("\n\n🛑 操作被用户中断。正在退出...")
        sys.exit(0)
    except Exception as e:
        print(f"\n\n💥 发生意外错误: {e}")

    input("\n任务已结束，请按 Enter 键退出。")